<?php
namespace My\Plugin\Console\Sqlfile\CliCommand;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\Database\DatabaseInterface;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputArgument;
use Joomla\Console\Command\AbstractCommand;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Filesystem\File;

class RunSqlfileCommand extends AbstractCommand
{
	/**
	 * The default command name
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected static $defaultName = 'sql:execute-file';
    
    /**
     * The params associated with the plugin, plus getter and setter
     * These are injected into this class by the plugin instance
     */
    protected $params;
    
    protected function getParams() {
        return $this->params;
    }
    
    public function setParams($params) {
        $this->params = $params;
    }

	/**
	 * Internal function to execute the command.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  integer  The command exit code
	 *
	 * @since   4.0.0
	 */
	protected function doExecute(InputInterface $input, OutputInterface $output): int
	{
		$symfonyStyle = new SymfonyStyle($input, $output);

		$symfonyStyle->title('Run a SQL file');
        
        // get the file of joomla sql statements, as an argument to the command
        $sqlfile = $input->getArgument('sqlfile');
        if (!file_exists($sqlfile)) {
            $symfonyStyle->error("{$sqlfile} does not exist");
            return false;
        }
        
        // get the file to log the actual sql statements, as an option to the command
        if ($logging = $input->getOption("logfile")) {
            $config = Factory::getApplication()->getConfig();
            $logpath = Factory::getApplication()->get('log_path', JPATH_ADMINISTRATOR . '/logs');
            // some users might enter an = after the "-l" option; if so we need to remove it
            $logfile = $logpath . '/' . ltrim($logging, "=");
        }
        
        // this is a standard option configured by Joomla
        $verbose = $input->getOption('verbose');
		
        // read the sql file into a buffer
        $buffer = file_get_contents($sqlfile);
        if ($buffer === false) {
            $symfonyStyle->error("Could not read contents of {$sqlfile}");
            return false;
        }
        
        // We reuse code from the Joomla install process in libraries/src/Installer/Installer.php
        $queries = Installer::splitSql($buffer);
        if (\count($queries) === 0) {
            $symfonyStyle->error("No SQL queries found in {$sqlfile}");
            return false;
        }

        $db = Factory::getContainer()->get(DatabaseInterface::class);
        
        // Get the plugin param defining whether we should use transaction control or not
        // Of course, some sql statements such as CREATE TABLE have implicit commits;
        // the code below doesn't really handle that situation.
        $transactionControl = $this->getParams()->get('txn', 1);
        if ($transactionControl) {
            $db->transactionStart();
        }
        
        foreach ($queries as $query) {

            try {
                if ($verbose) {
                    $symfonyStyle->info("Executing: \n{$query}");
                }
                $db->setQuery($query)->execute();
                $statement = $db->replacePrefix((string) $query);
                if ($logging) {
                    if (!File::append($logfile, $statement . "\n")) {
                        throw new \RuntimeException('Cannot write to log file.');
                    }
                }
                if ($verbose) {
                    $symfonyStyle->success(Text::_('Success'));
                }
            } catch (ExecutionFailureException $e) {
                if ($transactionControl) {
                    $db->transactionRollback();
                    $symfonyStyle->info("Rolling back database\n");
                }
                $symfonyStyle->warning($e->getMessage());
                return 2;  // or whatever error code you want to set
            }
        }
        if ($transactionControl) {
            $db->transactionCommit();
        }
        $symfonyStyle->success(\count($queries) . " SQL queries executed from {$sqlfile}");

		return 0;
	}

	/**
	 * Configure the command.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	protected function configure(): void
	{
		$this->addArgument('sqlfile', InputArgument::REQUIRED, 'file of joomla sql commands', null);
        $this->addOption('logfile', "l", InputOption::VALUE_REQUIRED, "log file");
        $this->setDescription('Run a list of SQL commands in a file.');
        $shortSynopsis = $this->getSynopsis(true);
		$this->setHelp(
			<<<EOF
The <info>%command.name%</info> command runs the SQL commands in the file passed as the --sqlfile argument
<info>php %command.full_name%</info>
Usage: {$shortSynopsis}
EOF
		);
	}
}